//package globalExamples;

import oracle.olapi.data.cursor.CompoundCursor;
import oracle.olapi.data.cursor.Cursor;
import oracle.olapi.data.cursor.CursorManager;
import oracle.olapi.data.cursor.ValueCursor;

import oracle.olapi.data.source.CursorManagerSpecification;
import oracle.olapi.data.source.CompoundCursorSpecification;
import oracle.olapi.data.source.CursorSpecification;
import oracle.olapi.data.source.DataProvider;
import oracle.olapi.data.source.Source;
import oracle.olapi.data.source.NumberSource;
import oracle.olapi.data.source.SpecifiedCursorManager;
import oracle.olapi.data.source.StringSource;
import oracle.olapi.data.source.ValueCursorSpecification;

import oracle.olapi.metadata.mdm.MdmAttribute;
import oracle.olapi.metadata.mdm.MdmHierarchy;
import oracle.olapi.metadata.mdm.MdmLevelHierarchy;
import oracle.olapi.metadata.mdm.MdmMeasure;
import oracle.olapi.metadata.mdm.MdmPrimaryDimension;

import oracle.olapi.transaction.TransactionProvider;
import oracle.olapi.transaction.NotCommittableException;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * Complete code for the examples in Chapter 10, Retrieving Query Results,
 * in the Oracle OLAP Developer's Guide to the  OLAP API.
 * 
 * This program uses the Context10g class, which uses the
 * CursorPrintWriter class.
 * 
 * @author Oracle Corporation
 */
public class RetrievingQueryResults 
{
private Context10g context = null;
private TransactionProvider tp = null;
private DataProvider dp = null;
private CursorPrintWriter cpw = null;

  public RetrievingQueryResults()
  {
  }

 public void run(String [] args)
  {
    context = new Context10g(args, false);

    // Get the TransactionProvider and the DataProvider.
    tp = context.getTransactionProvider();
    dp = context.getDataProvider();

    // Create a CursorPrintWriter.
    cpw = new CursorPrintWriter();
    
    // Create a query Source object to use in Example 10-1
    
    MdmMeasure mdmUnitPrice = context.getMdmMeasureByName("UNIT_PRICE");
    Source unitPrice = mdmUnitPrice.getSource();
    
    MdmPrimaryDimension mdmProdDim = 
                             context.getMdmPrimaryDimensionByName("PRODUCT");
    MdmPrimaryDimension mdmTimeDim = 
                             context.getMdmPrimaryDimensionByName("TIME");
    MdmLevelHierarchy mdmProdHier = (MdmLevelHierarchy) 
                                     mdmProdDim.getDefaultHierarchy();
    MdmLevelHierarchy mdmTimeHier = (MdmLevelHierarchy) 
                                     mdmTimeDim.getDefaultHierarchy();
    
    // Get the Source for the hierarchy.
    StringSource prodHier = (StringSource) mdmProdHier.getSource();
    StringSource timeHier = (StringSource) mdmTimeHier.getSource();

    Source prodSel = prodHier.selectValues(new String[] {
                                          "PRODUCT_ROLLUP::ITEM::13",
                                          "PRODUCT_ROLLUP::ITEM::14",
                                          "PRODUCT_ROLLUP::ITEM::15"});

    Source timeSel = timeHier.selectValues(new String[] {
                                          "CALENDAR::MONTH::55",
                                          "CALENDAR::MONTH::58",
                                          "CALENDAR::MONTH::61",
                                          "CALENDAR::MONTH::64"});

    Source querySource = unitPrice.join(prodSel).join(timeSel);

    // Call the methods that run the examples

    // Example 10-1
    creatingACursor(querySource);

    // Example 10-2
    ValueCursor prodValues = gettingASingleValue(mdmProdHier);

    // Example 10-3
    getAllValues(prodValues);

    // Example 10-4
    List sourcesFromEx4 = getValCursorsFromCompoundCursor(prodHier, timeHier);

    // Example 10-5
    getValsFormCompoundCursorWithNestedOutputs(sourcesFromEx4);

    // Example 10-6
    Source unitPriceByMonth = navigatingForATableView(unitPrice, 
                                                      timeHier, 
                                                      prodSel);

    // Example 10-7
    navigatingForACrossTabViewWOPages(unitPriceByMonth);
    
    // Example 10-8
      navigatingForACrossTabViewWPages(sourcesFromEx4);

    // Example 10-9
    CursorManagerSpecification cursorMngrSpec =
                          getCursorSpecsFromCursorManagerSpec(sourcesFromEx4); 
    // Example 10-10
    specifyingCalcExtent(cursorMngrSpec);
    
    // Example 10-11     
    specifyingCalcStartEndPositions(cursorMngrSpec);

    // Example 10-12
    calculatingTheSpan(sourcesFromEx4);

    // Example 10-13
    specifyingFetchSize(cursorMngrSpec);
  }

  // Example 10-1
  private void creatingACursor(Source querySource)
  {
    context.println("Example 10-1, Creating a Cursor");

    try
    {
      tp.prepareCurrentTransaction();
    }
    catch(NotCommittableException e){
      System.out.println("Cannot commit the current Transaction. " + e);
    }
    tp.commitCurrentTransaction();
    CursorManagerSpecification cursorMngrSpec =
                     dp.createCursorManagerSpecification(querySource);
    SpecifiedCursorManager cursorMngr = 
                               dp.createCursorManager(cursorMngrSpec);
    Cursor queryCursor = cursorMngr.createCursor();

    // ... Use the Cursor in some way, such as to display its values.
    context.displayCursor(queryCursor);

    cursorMngr.close();
  }

  // Example 10-2
  private ValueCursor gettingASingleValue(MdmHierarchy mdmProdHier)
  {
    context.println("\nExample 10-2, Getting a Single Value from a ValueCursor");

    Source prodSource = mdmProdHier.getSource();
    // Because prodSource is a primary Source, you do not need to
    // prepare and commit the current Transaction.
    CursorManagerSpecification cursorMngrSpec =
                   dp.createCursorManagerSpecification(prodSource);
    SpecifiedCursorManager cursorMngr = 
                               dp.createCursorManager(cursorMngrSpec);
    Cursor prodCursor = cursorMngr.createCursor();
    // Cast the Cursor to a ValueCursor.
    ValueCursor prodValues = (ValueCursor) prodCursor;
    // Set the position to the fifth element of the ValueCursor.
    prodValues.setPosition(5);
    
    // Product values are strings. Get the String value at the current
    // position.
    String value = prodValues.getCurrentString();
    
    // Do something with the value, such as display it...
    context.println(value);
    // Close the SpecifiedCursorManager.

    return prodValues;
    
    //cursorMngr.close();
  }

  //Example 10-3
  private void getAllValues(ValueCursor prodValues)
  {
    context.println("\nExample 10-3, Getting All of the Values from a ValueCursor");

    // prodValues is the ValueCursor for prodSource
    prodValues.setPosition(1);
    do
    {
       System.out.println(prodValues.getCurrentValue());
    }while(prodValues.next());
  }

  // Example 10-4
  private List getValCursorsFromCompoundCursor(StringSource prodHier,
                                               StringSource timeHier)
  {
    context.println("\nExample 10-4, Getting ValueCursor Objects from a " +
                                "CompoundCursor example.");

    // Get the metadata objects.
    MdmMeasure mdmUnits = context.getMdmMeasureByName("UNITS");
    NumberSource units = (NumberSource) mdmUnits.getSource();
    
    MdmPrimaryDimension mdmChanDim = 
                             context.getMdmPrimaryDimensionByName("CHANNEL");
    MdmPrimaryDimension mdmCustDim = 
                             context.getMdmPrimaryDimensionByName("CUSTOMER");
    MdmLevelHierarchy mdmChanHier = (MdmLevelHierarchy) 
                                     mdmChanDim.getDefaultHierarchy();
    MdmLevelHierarchy mdmCustHier = (MdmLevelHierarchy) 
                                     mdmCustDim.getDefaultHierarchy();
        
    StringSource chanHier = (StringSource) mdmChanHier.getSource();
    StringSource custHier = (StringSource) mdmCustHier.getSource();

    Source chanSel = chanHier.selectValue("CHANNEL_ROLLUP::CHANNEL::2");

    Source custSel = custHier.selectValues(new String[] {
                                           "SHIPMENTS_ROLLUP::SHIP_TO::58",
                                           "SHIPMENTS_ROLLUP::SHIP_TO::61",
                                           "SHIPMENTS_ROLLUP::SHIP_TO::65"});

    Source prodSel = prodHier.selectValues(new String[] {
                                          "PRODUCT_ROLLUP::ITEM::13",
                                          "PRODUCT_ROLLUP::ITEM::14",
                                          "PRODUCT_ROLLUP::ITEM::15"});

    Source timeSel = timeHier.selectValues(new String[] {
                                          "CALENDAR::MONTH::43",
                                          "CALENDAR::MONTH::44",
                                          "CALENDAR::MONTH::45"});

    // Code from the example
    Source unitsForSelections = units.join(prodSel)
                                     .join(custSel)
                                     .join(timeSel)
                                     .join(chanSel);
    // Prepare and commit the current Transaction
    try
    {
      tp.prepareCurrentTransaction();
    }
    catch(NotCommittableException e)
    {
      context.println("Cannot commit the current Transaction. " + e);
    }
    tp.commitCurrentTransaction();
    
    // Create a Cursor for salesForSelections
    CursorManagerSpecification cursorMngrSpec =
                dp.createCursorManagerSpecification(unitsForSelections);
    SpecifiedCursorManager cursorMngr = 
                                 dp.createCursorManager(cursorMngrSpec);
    CompoundCursor unitsForSelCursor = (CompoundCursor) 
                                        cursorMngr.createCursor();
    
    // Cast salesForSelCursor to a CompoundCursor
    //CompoundCursor salesCompndCrsr = (CompoundCursor) salesValues;
    
    // Get the base ValueCursor
    ValueCursor specifiedUnitsVals = unitsForSelCursor.getValueCursor();
    
    // Get the outputs
    List outputs = unitsForSelCursor.getOutputs();
    ValueCursor chanSelVals = (ValueCursor) outputs.get(0);
    ValueCursor timeSelVals = (ValueCursor) outputs.get(1);
    ValueCursor custSelVals = (ValueCursor) outputs.get(2);
    ValueCursor prodSelVals = (ValueCursor) outputs.get(3);

    // You can now get the values from the ValueCursor objects.
    // When you have finished using the Cursor objects, close the
    // SpecifiedCursorManager.
    int i = 1;
    do 
    {
      context.println(i + ": " + chanSelVals.getCurrentString() + ", " +
                      timeSelVals.getCurrentString() + ", " +
                      custSelVals.getCurrentString() + ", " +
                      prodSelVals.getCurrentString() + ", " +
                      specifiedUnitsVals.getCurrentValue());
      i++;
    }while(unitsForSelCursor.next());
    
    cursorMngr.close();

    // Create an ArrayList of the Source objects for use by other examples.
    ArrayList sourcesFromExample4 = new ArrayList();
    sourcesFromExample4.add(units);
    sourcesFromExample4.add(prodSel);
    sourcesFromExample4.add(custSel);
    sourcesFromExample4.add(timeSel);
    sourcesFromExample4.add(chanSel);

    return sourcesFromExample4;
  }

  // Example 10-5
  private void getValsFormCompoundCursorWithNestedOutputs(List sourcesFromEx4)
  {
    context.println("\nExample 10-5, Getting Values from a CompoundCursor " +
                                "with Nested Outputs example.");
                                
    Source units = (Source) sourcesFromEx4.get(0);
    Source prodSel = (Source) sourcesFromEx4.get(1);
    Source custSel = (Source) sourcesFromEx4.get(2);
    Source timeSel = (Source) sourcesFromEx4.get(3);
    Source chanSel = (Source) sourcesFromEx4.get(4);

    // Code from the example
    Source custByChanSel = custSel.join(chanSel);
    Source unitsForSelections = units.join(prodSel)
                                     .join(timeSel)
                                     .join(custByChanSel);

    // Prepare and commit the current Transaction
    try
    {
      tp.prepareCurrentTransaction();
    }
    catch(NotCommittableException e)
    {
      System.out.println("Cannot prepare the current Transaction. " + e);
    }
    tp.commitCurrentTransaction();
    
    // Create a Cursor for unitsForSelections
    CursorManagerSpecification cursorMngrSpec =
                dp.createCursorManagerSpecification(unitsForSelections);
    SpecifiedCursorManager cursorMngr = 
                                 dp.createCursorManager(cursorMngrSpec);
    Cursor unitsForSelCursor = cursorMngr.createCursor();
    
    // Send the Cursor to a method that does different operations 
    // depending on whether the Cursor is a CompoundCursor or a 
    // ValueCursor.
    printCursor(unitsForSelCursor);
    cursorMngr.close();
    // ...the remaining code of someMethod...
  }
    
  // The printCursor method has a do...while loop that moves through the positions
  // of the Cursor passed to it. At each position, the method prints the number of
  //  the iteration through the loop and then a colon and a space. The output
  // object is a PrintWriter. The method calls the private _printTuple method and
  // then prints a new line. A "tuple" is the set of output ValueCursor values
  // specified by one position of the parent CompoundCursor. The method prints one
  // line for each position of the parent CompoundCursor.
  private void printCursor(Cursor rootCursor) 
  {  
    int i = 1;
    do 
    {
       cpw.print(i++ + ": ");
         _printTuple(rootCursor);
         cpw.print("\n");
         cpw.flush();
    }while(rootCursor.next());
  }
    
  // If the Cursor passed to the _printTuple method is a ValueCursor, 
  // the method prints the value at the current position of the ValueCursor. 
  // If the Cursor passed in is a CompoundCursor, the method gets the 
  // outputs of the CompoundCursor and iterates through the outputs,  
  // recursively calling itself for each output. The method then gets the 
  // base ValueCursor of the CompoundCursor and calls itself again. 
  private void _printTuple(Cursor cursor) 
  {
    if(cursor instanceof CompoundCursor) 
    {
      CompoundCursor compoundCursor = (CompoundCursor)cursor;
      // Put an open parenthesis before the value of each output
      cpw.print("(");
      Iterator iterOutputs = compoundCursor.getOutputs().iterator();
      Cursor output = (Cursor)iterOutputs.next();
      _printTuple(output);
      while(iterOutputs.hasNext()) 
      {
        // Put a comma after the value of each output
        cpw.print(",");
        _printTuple((Cursor)iterOutputs.next());
      }
      // Put a comma after the value of the last output
      cpw.print(",");
      // Get the base ValueCursor
      _printTuple(compoundCursor.getValueCursor());
        
      // Put a close parenthesis after the base value to indicate  
      // the end of the tuple.
      cpw.print(")");
    }
    else if(cursor instanceof ValueCursor) 
    {
      ValueCursor valueCursor = (ValueCursor) cursor;
      if (valueCursor.hasCurrentValue())
        cpw.print(valueCursor.getCurrentValue());
      else                       // If this position has a null value
        cpw.print("NA");
    }
  }

  // Example 10-6
  private Source navigatingForATableView(Source unitPrice, 
                                         StringSource timeHier, 
                                         Source prodSel)
  {
    context.println("\nExample 10-6, Navigating For a Table View\n");

    // Select the first month of the calendar quarters for 2001, 2002
    Source timeSel = timeHier.selectValues(new String[] {
                                           "CALENDAR::MONTH::55",
                                           "CALENDAR::MONTH::58",
                                           "CALENDAR::MONTH::61",
                                           "CALENDAR::MONTH::64",
                                           "CALENDAR::MONTH::69",
                                           "CALENDAR::MONTH::72",
                                           "CALENDAR::MONTH::75",
                                           "CALENDAR::MONTH::78"});
    Source unitPriceByMonth = unitPrice.join(prodSel)
                                       .join(timeSel);
    try{
      tp.prepareCurrentTransaction();
    }
    catch(NotCommittableException e){
      cpw.println("Cannot prepare the current Transaction. " + e);
    }
    tp.commitCurrentTransaction();
    
    // Create a Cursor for unitPriceByMonth
    CursorManagerSpecification cursorMngrSpec =
                dp.createCursorManagerSpecification(unitPriceByMonth);
    SpecifiedCursorManager cursorMngr = 
                                 dp.createCursorManager(cursorMngrSpec);
    Cursor unitPriceByMonthCursor = cursorMngr.createCursor();
    
    // Cast the Cursor to a CompoundCursor
    CompoundCursor rootCursor = (CompoundCursor) unitPriceByMonthCursor;
    
    // Determine a starting position and the number of rows to display
    int start = 7;
    int numRows = 12;

    cpw.println("Month     Product     Unit Price");
    cpw.println("-----     -------     ----------");
    
    // Iterate through the specified positions of the root CompoundCursor.
    // Assume that the Cursor contains at least (start + numRows) positions.
    for(int pos = start; pos < start + numRows; pos++) 
    {
      // Set the position of the root CompoundCursor
      rootCursor.setPosition(pos);
       
      // Print the local values of the output and base ValueCursors.
      // The getLocalValue method gets the local value from the unique
      // value of a dimension element.
      String timeValue = ((ValueCursor)rootCursor.getOutputs().get(0))
                         .getCurrentString();
      String timeLocVal = getLocalValue(timeValue);
      String prodValue = ((ValueCursor)rootCursor.getOutputs().get(1))
                         .getCurrentString();
      String prodLocVal = getLocalValue(prodValue);
      Object price = rootCursor.getValueCursor().getCurrentValue();
      cpw.println("  " + timeLocVal + "         " + prodLocVal  
                  + "        " +  price);
    }
    cursorMngr.close();

    return unitPriceByMonth;
  }

  // Example 10-7
  private void navigatingForACrossTabViewWOPages(Source unitPriceByMonth)
  {
    context.println("\nExample 10-7, Navigating For a Crosstab View Without Pages");

    //Source unitPriceByMonth = unitPrice.join(prodSel)
    //                                   .join(timeSel);
    
    try
    {
      tp.prepareCurrentTransaction();
    }
    catch(NotCommittableException e)
    {
      System.out.println("Cannot prepare the current Transaction. " + e);
    }
    tp.commitCurrentTransaction();
    
    // Create a Cursor for unitPriceByMonth
    CursorManagerSpecification cursorMngrSpec =
                dp.createCursorManagerSpecification(unitPriceByMonth);
    SpecifiedCursorManager cursorMngr = 
                             dp.createCursorManager(cursorMngrSpec);
    CompoundCursor rootCursor = (CompoundCursor) cursorMngr.createCursor();
       
    // Get the outputs and the ValueCursor.
    List outputs = rootCursor.getOutputs();
    // The first output has the values of timeSel, the slower varying output.
    ValueCursor rowCursor = (ValueCursor) outputs.get(0);
    // The second output has the faster varying values of prodSel.
    ValueCursor columnCursor = (ValueCursor) outputs.get(1);
    // The base ValueCursor has the values from unitPrice
    ValueCursor unitPriceValues = rootCursor.getValueCursor();

    // Display the values as a crosstab.
    cpw.println("\t        Product");
    cpw.println("\t-----------------------");
    cpw.print("Month");
    do 
    {
      String value = ((ValueCursor) columnCursor).getCurrentString();
      cpw.print("\t" + getLocalValue(value) + "  ");
    } while (columnCursor.next());
    cpw.println();
    cpw.println("-----\t-------\t-------\t-------");
    
    // Reset the column Cursor to its first element.
    columnCursor.setPosition(1);

    do 
    {
      // Print the row dimension values.
      String value = ((ValueCursor) rowCursor).getCurrentString();
      cpw.print(getLocalValue(value) + "\t");
      // Loop over columns
      do 
      {
        // Print data value
        cpw.print(unitPriceValues.getCurrentValue() + "\t");
      } while (columnCursor.next());

      cpw.println();
  
      // Reset the column Cursor to its first element.
      columnCursor.setPosition(1);
    } while (rowCursor.next());
  
    cursorMngr.close();
  }

  // Example 10-8
  private void navigatingForACrossTabViewWPages(List sourcesFromEx4)
  {
    context.println("\nExample 10-8, Navigating For a Crosstab View With Pages");

    Source units = (Source) sourcesFromEx4.get(0);
    Source prodSel = (Source) sourcesFromEx4.get(1);
    Source custSel = (Source) sourcesFromEx4.get(2);
    Source timeSel = (Source) sourcesFromEx4.get(3);
    Source chanSel = (Source) sourcesFromEx4.get(4);

    Source unitsForSelections = units.join(prodSel)
                                     .join(custSel)
                                     .join(timeSel)
                                     .join(chanSel);
    // Prepare and commit the current Transaction
    try{
      tp.prepareCurrentTransaction();
    }
    catch(NotCommittableException e){
      context.println("Cannot prepare the current Transaction. " + e);
    }
    tp.commitCurrentTransaction();
    
    // Create a Cursor for unitsForSelections
    CursorManagerSpecification cursorMngrSpec =
                dp.createCursorManagerSpecification(unitsForSelections);
    SpecifiedCursorManager cursorMngr = 
                                 dp.createCursorManager(cursorMngrSpec);
    CompoundCursor unitsForSelCursor = (CompoundCursor) 
                                        cursorMngr.createCursor();

    // Display the results in a crosstab
    printAsCrosstab(unitsForSelCursor);
    cursorMngr.close();
  }

  protected void printAsCrosstab(CompoundCursor rootCursor) 
  {
    List outputs = rootCursor.getOutputs();
    int nOutputs = outputs.size();

    // Set the initial positions of all outputs
    Iterator outputIter = outputs.iterator();
    while (outputIter.hasNext())
      ((Cursor) outputIter.next()).setPosition(1);
  
    // The last output is fastest-varying; it represents columns.
    // The next to last output represents rows.
    // All other outputs are on the page.
    Cursor colCursor = (Cursor) outputs.get(nOutputs - 1);
    Cursor rowCursor = (Cursor) outputs.get(nOutputs - 2);
    ArrayList pageCursors = new ArrayList();
    for (int i = 0 ; i < nOutputs - 2 ; i++) 
    {
      pageCursors.add(outputs.get(i));
    }

    // Get the base ValueCursor, which has the data values
    ValueCursor dataCursor = rootCursor.getValueCursor();

    // Print the pages of the crosstab
    printPages(pageCursors, 0, rowCursor, colCursor, dataCursor);
  }

  // Prints the pages of a crosstab
  private void printPages(List pageCursors, int pageIndex, Cursor rowCursor,
                          Cursor colCursor, ValueCursor dataCursor) 
  {
    // Get a Cursor for this page
    Cursor pageCursor = (Cursor) pageCursors.get(pageIndex);
  
    // Loop over the values of this page dimension
    do 
    {
      // If this is the fastest-varying page dimension, print a page
      if (pageIndex == pageCursors.size() - 1) 
      {
      	// Print the values of the page dimensions
        printPageHeadings(pageCursors);
  
        // Print the column headings
        printColumnHeadings(colCursor);
  
        // Print the rows
      	printRows(rowCursor, colCursor, dataCursor);
  
       	// Print a couple of blank lines to delimit pages
        cpw.println();
        cpw.println();
      }
  
      // If this is not the fastest-varying page, recurse to the
      // next fastest varying dimension.
      else 
      {
  	    printPages(pageCursors, pageIndex + 1, rowCursor, colCursor, 
                   dataCursor);
  	  }
    } while (pageCursor.next());
  
    // Reset this page dimension Cursor to its first element.
    pageCursor.setPosition(1);
  }
  
  // Prints the values of the page dimensions on each page
  private void printPageHeadings(List pageCursors)
  {
    // Print the values of the page dimensions
    Iterator pageIter = pageCursors.iterator();
    while (pageIter.hasNext())
    {
      String value = ((ValueCursor) pageIter.next()).getCurrentString();    
      cpw.println(getLocalValue(value));
    }
    cpw.println();
  }
  
  // Prints the column headings on each page
  private void printColumnHeadings(Cursor colCursor) 
  {
    do 
    {
      cpw.print("\t");
      String value = ((ValueCursor) colCursor).getCurrentString();
      cpw.print(getLocalValue(value));
    } while (colCursor.next());
    cpw.println();
    colCursor.setPosition(1);
  }
  
  // Prints the rows of each page
  private void printRows(Cursor rowCursor, Cursor colCursor,
                         ValueCursor dataCursor) 
  {
    // Loop over rows
    do 
    {
      // Print row dimension value
      String value = ((ValueCursor) rowCursor).getCurrentString();
      cpw.print(getLocalValue(value));
      cpw.print("\t");
      // Loop over columns
      do 
      {
        // Print data value
        cpw.print(dataCursor.getCurrentValue());
        cpw.print("\t");
  	    } while (colCursor.next());

      cpw.println();
  
      // Reset the column Cursor to its first element
      colCursor.setPosition(1);
    } while (rowCursor.next());
  
    // Reset the row Cursor to its first element
    rowCursor.setPosition(1);
  }

  private String getLocalValue(String uniqueValue)
  {
    int index = 0;
    if (uniqueValue.indexOf("::") > 0)
    {
      index = uniqueValue.lastIndexOf("::");
      String localValue = uniqueValue.substring((uniqueValue.lastIndexOf("::")
                                                  + 2),
                                                 uniqueValue.length());
      return(localValue);
    }
    else
      return uniqueValue;
  }

  // Example 10-9
  private CursorManagerSpecification
                       getCursorSpecsFromCursorManagerSpec(List sourcesFromEx4)
  {
    context.println("\nExample 10-9, Getting CursorSpecification Objects " +
                    "from a CursorManagerSpecification\n");

    context.println("This example has no text output. It returns a " +
                    "CursorManagerSpecification used by other examples.\n");
    
    Source units  = (Source) sourcesFromEx4.get(0);
    Source prodSel = (Source) sourcesFromEx4.get(1);
    Source custSel = (Source) sourcesFromEx4.get(2);
    Source timeSel = (Source) sourcesFromEx4.get(3);
    Source chanSel = (Source) sourcesFromEx4.get(4);

    Source unitsForSelections = units.join(prodSel)
                                     .join(custSel)
                                     .join(timeSel)
                                     .join(chanSel);
                                     
    // Prepare and commit the current Transaction
    try
    {
      tp.prepareCurrentTransaction();
    }
    catch(NotCommittableException e)
    {
      context.println("Cannot prepare the current Transaction. " + e);
    }
    tp.commitCurrentTransaction();
    
    // Create a Cursor for unitsForSelections
    CursorManagerSpecification cursorMngrSpec =
         dp.createCursorManagerSpecification(unitsForSelections);
    
    // Get the root CursorSpecification of the CursorManagerSpecification.
    CompoundCursorSpecification rootCursorSpec =
    (CompoundCursorSpecification) cursorMngrSpec.getRootCursorSpecification();
    
    // Get the CursorSpecification for the base values
    ValueCursorSpecification baseValueSpec =
                            rootCursorSpec.getValueCursorSpecification();
    
    // Get the CursorSpecification objects for the outputs
    List outputSpecs = rootCursorSpec.getOutputs();
    ValueCursorSpecification chanSelValCSpec = 
                           (ValueCursorSpecification) outputSpecs.get(0);
    ValueCursorSpecification timeSelValCSpec = 
                           (ValueCursorSpecification) outputSpecs.get(1);
    ValueCursorSpecification prodSelValCSpec = 
                           (ValueCursorSpecification) outputSpecs.get(2);
    ValueCursorSpecification custSelValCSpec = 
                           (ValueCursorSpecification) outputSpecs.get(3);

    return cursorMngrSpec;
  }

  // Example 10-10
  private void specifyingCalcExtent(CursorManagerSpecification cursorMngrSpec)
  {
    context.println("\nExample 10-10, Specifying the Calculation of the Extent " 
                     + "of a Cursor\n");

    CompoundCursorSpecification rootCursorSpec = (CompoundCursorSpecification)
                                   cursorMngrSpec.getRootCursorSpecification();
    rootCursorSpec.setExtentCalculationSpecified(true);

    boolean isSet = rootCursorSpec.isExtentCalculationSpecified();
    isCalcSet(isSet, 1);
  }

  private void isCalcSet(boolean isSet, int type)
  {
    if(isSet)
    {
      switch(type)
      {
        case 1: // Extent
          context.println("The calculation of the extent is set on the " +
                    "CursorSpecification.");
          break;
        case 2: // Parent start
          context.println("The calculation of the starting position of the " +
                           "current child Cursor");
          context.println("in its parent Cursor is set on the " +
                          "CursorSpecification.");
          break;
        case 3: // Parent end
          context.println("The calculation of the ending position of the " +
                           "current child Cursor");
          context.println("in its parent Cursor is set on the " +
                          "CursorSpecification.");
          break;
        default:  // not a valid value for type
          context.println("The calculation type is incorrect.");
      }
    }
    else
    {
      context.println("The calculation is not set on the CursorSpecification.");
    }
  }

  // Example 10-11
  private void specifyingCalcStartEndPositions(CursorManagerSpecification 
                                               cursorMngrSpec)
  {
    context.println("\nExample 10-11, Specifying the Calculation of  " +
                    "Starting and Ending Positions in a Parent\n");
                                  
    // Specifying the Calculation of Starting and Ending Positions in a Parent 
    CompoundCursorSpecification rootCursorSpec = (CompoundCursorSpecification) 
                                  cursorMngrSpec.getRootCursorSpecification();

    // Get the List of CursorSpecification objects for the outputs. 
    // Iterate through the list, specifying the calculation of the extent
    // for each output CursorSpecification. 
    Iterator iterOutputSpecs = rootCursorSpec.getOutputs().iterator();
    
    while(iterOutputSpecs.hasNext())
    {
      ValueCursorSpecification valCursorSpec = (ValueCursorSpecification)
                                                iterOutputSpecs.next();
      valCursorSpec.setParentStartCalculationSpecified(true);
      valCursorSpec.setParentEndCalculationSpecified(true);
    }

    boolean isSet;
    Iterator outputSpecsItr = rootCursorSpec.getOutputs().iterator();
    while(outputSpecsItr.hasNext())
    {
      ValueCursorSpecification valCursorSpec = (ValueCursorSpecification)
                                                  outputSpecsItr.next();
      isSet = valCursorSpec.isParentStartCalculationSpecified();
      isCalcSet(isSet, 2); // added to verify that calculation is set
      isSet = valCursorSpec.isParentEndCalculationSpecified();
      isCalcSet(isSet, 3); // added to verify that calculation is set
    }
    // Create a Cursor and display the results to show the parent
    // starting and ending positions
    SpecifiedCursorManager cursorManager =
		                    dp.createCursorManager(cursorMngrSpec);
                        
    Cursor cursor = cursorManager.createCursor();
    context.displayCursor(cursor);
	  cursorManager.close(); 
  }

  // Example 10-12
  public void calculatingTheSpan(List sourcesFromEx4)
  {
    context.println("\nExample 10-12, Calculating the Span of the Positions " +
                    "in the Parent of a Value\n");

    Source units  = (Source) sourcesFromEx4.get(0);
    Source prodSel = (Source) sourcesFromEx4.get(1);
    Source custSel = (Source) sourcesFromEx4.get(2);
    Source timeSel = (Source) sourcesFromEx4.get(3);
    Source chanSel = (Source) sourcesFromEx4.get(4);

    Source unitsForSelections = units.join(prodSel)
                                     .join(custSel)
                                     .join(timeSel)
                                     .join(chanSel);
                                     
    // Prepare and commit the current Transaction
    try
    {
      tp.prepareCurrentTransaction();
    }
    catch(NotCommittableException e)
    {
      cpw.println("Cannot prepare the current Transaction. " + e);
    }
    tp.commitCurrentTransaction();
    
    // Create a Cursor for unitsForSelections
    CursorManagerSpecification cursorMngrSpec =
         dp.createCursorManagerSpecification(unitsForSelections);
    
    // Get the root CursorSpecification of the CursorManagerSpecification.
    CompoundCursorSpecification rootCursorSpec = (CompoundCursorSpecification)
                                    cursorMngrSpec.getRootCursorSpecification();

    // Get the CursorSpecification objects for the outputs
    List outputSpecs = rootCursorSpec.getOutputs();
    ValueCursorSpecification timeSelValCSpec = 
              (ValueCursorSpecification) outputSpecs.get(3); // output for time
    ValueCursorSpecification prodSelValCSpec = 
          (ValueCursorSpecification) outputSpecs.get(1);  // output for product
    
    // Specify the calculation of the starting and ending positions
    timeSelValCSpec.setParentStartCalculationSpecified(true);
    timeSelValCSpec.setParentEndCalculationSpecified(true);
    prodSelValCSpec.setParentStartCalculationSpecified(true);
    prodSelValCSpec.setParentEndCalculationSpecified(true);
    
    // Create the CursorManager and the Cursor
    SpecifiedCursorManager cursorMngr = dp.createCursorManager(cursorMngrSpec);
    CompoundCursor rootCursor = (CompoundCursor) cursorMngr.createCursor();
    
    // Get the child Cursor objects
    ValueCursor baseValCursor = rootCursor.getValueCursor();
    List outputs = rootCursor.getOutputs();
    ValueCursor chanSelVals = (ValueCursor) outputs.get(0);
    ValueCursor timeSelVals = (ValueCursor) outputs.get(1);
    ValueCursor custSelVals = (ValueCursor) outputs.get(2);
    ValueCursor prodSelVals = (ValueCursor) outputs.get(3);
        
    // Set the position of the root CompoundCursor
    rootCursor.setPosition(15);
    
    // Get the values at the current position and determine the span
    // of the values of the time and product outputs.
    cpw.print(chanSelVals.getCurrentValue() + ", ");
    cpw.print(timeSelVals.getCurrentValue() + ", ");
    cpw.print(custSelVals.getCurrentValue() + ", ");
    cpw.print(prodSelVals.getCurrentValue() + ", ");
    cpw.println(baseValCursor.getCurrentValue());
    
    // Determine the span of the values of the two fastest varying outputs
    long span;
    //int parentStart = prodSelVals.getParentStart();
    //int parentEnd = prodSelVals.getParentEnd();
   // if (parentEnd == parentStart)
   //   span = 1;
    //else
    //  span = (parentEnd - parentStart) -1;
    span = ((prodSelVals.getParentEnd() - prodSelVals.getParentStart()) +1);
    cpw.println("The span of " + prodSelVals.getCurrentValue() +
                " at the current position is " + span + ".");
    span = ((timeSelVals.getParentEnd() - timeSelVals.getParentStart()) +1);
    cpw.println("The span of " + timeSelVals.getCurrentValue() +
                   " at the current position is " + span + ".");

    cursorMngr.close();
  }

  // Example 10-13
  private void specifyingFetchSize(CursorManagerSpecification cursorMngrSpec)
  {
    context.println("\nExample 10-13, Specifying a Fetch Size\n");

    // Get the root CursorSpecification
    CursorSpecification rootCursorSpec = 
                                   cursorMngrSpec.getRootCursorSpecification();
    // Get the default fetch size
    context.println("The default fetch size is "
                    + rootCursorSpec.getDefaultFetchSize() + ".");
    CursorManager cursorMngr = dp.createCursorManager(cursorMngrSpec);
    Cursor rootCursor = cursorMngr.createCursor();
    rootCursor.setFetchSize(10);
    context.println("The fetch size is now " + rootCursor.getFetchSize() + ".");
  }

  public static void main(String[] args)
  {
    new RetrievingQueryResults().run(args);
  }
}
